<?php
// app/Http/Controllers/ChatPushController.php (moved from Api namespace)

namespace App\Http\Controllers;

use App\Models\PushSubscription;
use App\Services\ChatPushNotificationService;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Log;
use Illuminate\Validation\ValidationException;

class ChatPushController extends Controller
{
    private ChatPushNotificationService $pushService;

    public function __construct(ChatPushNotificationService $pushService)
    {
        $this->pushService = $pushService;
    }

    /**
     * Get current user identifier (same as ChatController)
     */
    private function getCurrentUserIdentifier(Request $request): array
    {
        try {
            $user = $request->attributes->get('authenticated_user');
            
            if (!$user) {
                Log::error('No authenticated user found in request attributes');
                abort(401, 'User not authenticated');
            }

            if (get_class($user) === 'App\Models\Admin') {
                return [
                    'type' => 'admin',
                    'id' => $user->id,
                    'user' => $user
                ];
            }
            
            // Handle employee
            if (isset($user->employee_id)) {
                return [
                    'type' => 'employee',
                    'id' => $user->employee_id,
                    'user' => $user
                ];
            }
            
            Log::error('Unable to determine user type', ['user_class' => get_class($user)]);
            abort(403, 'Unable to determine user identity.');
        } catch (\Exception $e) {
            Log::error('Error getting current user identifier', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            abort(500, 'Internal server error');
        }
    }

    /**
     * Get current tenant ID (same as ChatController)
     */
    private function getCurrentTenantId(Request $request): int
    {
        $tenantId = $request->attributes->get('tenant_id');
        
        if (!$tenantId) {
            Log::error('No tenant_id found in request attributes');
            abort(400, 'Tenant not found');
        }
        
        return $tenantId;
    }

    /**
     * Subscribe user to push notifications
     */
    public function subscribe(Request $request): JsonResponse
    {
        try {
            // Validate the request
            $request->validate([
                'endpoint' => 'required|url',
                'keys.p256dh' => 'required|string',
                'keys.auth' => 'required|string',
            ]);

            $userIdentifier = $this->getCurrentUserIdentifier($request);
            $tenantId = $this->getCurrentTenantId($request);
            
            Log::info('Push subscription request', [
                'user_type' => $userIdentifier['type'],
                'user_id' => $userIdentifier['id'],
                'tenant_id' => $tenantId,
                'endpoint' => $request->endpoint
            ]);
            
            $subscriptionData = [
                'tenant_id' => $tenantId,
                'endpoint' => $request->endpoint,
                'public_key' => $request->input('keys.p256dh'),
                'auth_token' => $request->input('keys.auth'),
                'content_encoding' => $request->input('contentEncoding', 'aesgcm'),
            ];

            if ($userIdentifier['type'] === 'admin') {
                $subscriptionData['admin_id'] = $userIdentifier['id'];
                $subscriptionData['employee_id'] = null;
            } else {
                $subscriptionData['employee_id'] = $userIdentifier['id'];
                $subscriptionData['admin_id'] = null;
            }

            // Create or update subscription
            PushSubscription::updateOrCreate(
                [
                    'tenant_id' => $tenantId,
                    'employee_id' => $subscriptionData['employee_id'],
                    'admin_id' => $subscriptionData['admin_id'],
                    'endpoint' => $request->endpoint,
                ],
                $subscriptionData
            );

            Log::info('Push subscription saved successfully', [
                'user_type' => $userIdentifier['type'],
                'user_id' => $userIdentifier['id']
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Push notification subscription saved successfully'
            ]);
            
        } catch (ValidationException $e) {
            Log::warning('Push subscription validation failed', [
                'errors' => $e->errors()
            ]);
            
            return response()->json([
                'success' => false,
                'error' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);
            
        } catch (\Exception $e) {
            Log::error('Push subscription failed', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'success' => false,
                'error' => 'Failed to save push subscription',
                'message' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Unsubscribe user from push notifications
     */
    public function unsubscribe(Request $request): JsonResponse
    {
        try {
            $request->validate([
                'endpoint' => 'required|url',
            ]);

            $userIdentifier = $this->getCurrentUserIdentifier($request);
            $tenantId = $this->getCurrentTenantId($request);

            $query = PushSubscription::where('tenant_id', $tenantId)
                                    ->where('endpoint', $request->endpoint);

            if ($userIdentifier['type'] === 'admin') {
                $query->where('admin_id', $userIdentifier['id']);
            } else {
                $query->where('employee_id', $userIdentifier['id']);
            }

            $deleted = $query->delete();

            Log::info('Push subscription removed', [
                'user_type' => $userIdentifier['type'],
                'user_id' => $userIdentifier['id'],
                'deleted_count' => $deleted
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Push notification subscription removed successfully'
            ]);
            
        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'error' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);
            
        } catch (\Exception $e) {
            Log::error('Push unsubscription failed', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'success' => false,
                'error' => 'Failed to remove push subscription'
            ], 500);
        }
    }

    /**
     * Get subscription status
     */
    public function status(Request $request): JsonResponse
    {
        try {
            $userIdentifier = $this->getCurrentUserIdentifier($request);
            $tenantId = $this->getCurrentTenantId($request);

            $query = PushSubscription::where('tenant_id', $tenantId);

            if ($userIdentifier['type'] === 'admin') {
                $query->where('admin_id', $userIdentifier['id']);
            } else {
                $query->where('employee_id', $userIdentifier['id']);
            }

            $hasSubscription = $query->exists();
            $vapidPublicKey = config('app.vapid_public_key');
            
            if (!$vapidPublicKey) {
                Log::error('VAPID public key not configured');
                return response()->json([
                    'success' => false,
                    'error' => 'Push notifications not properly configured on server'
                ], 500);
            }

            Log::info('Push subscription status retrieved', [
                'user_type' => $userIdentifier['type'],
                'user_id' => $userIdentifier['id'],
                'subscribed' => $hasSubscription,
                'has_vapid_key' => !empty($vapidPublicKey)
            ]);

            return response()->json([
                'success' => true,
                'subscribed' => $hasSubscription,
                'vapid_public_key' => $vapidPublicKey
            ]);
            
        } catch (\Exception $e) {
            Log::error('Failed to get push subscription status', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'success' => false,
                'error' => 'Failed to get subscription status',
                'message' => $e->getMessage()
            ], 500);
        }
    }
}