<?php

namespace App\Http\Controllers;

use App\Models\Admin;
use App\Models\Conversation;
use App\Models\ConversationParticipant;
use App\Models\Employee;
use App\Models\Message;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class ChatNotificationController extends Controller
{
    /**
     * Get the current authenticated user's identifier
     */
    private function getCurrentUserIdentifier(Request $request)
    {
        $user = $request->attributes->get('authenticated_user');
        
        if (get_class($user) === 'App\Models\Admin') {
            return [
                'type' => 'admin',
                'id' => $user->id,
                'user' => $user
            ];
        }
        
        // Handle employee
        if (isset($user->employee_id)) {
            return [
                'type' => 'employee',
                'id' => $user->employee_id,
                'user' => $user
            ];
        }
        
        abort(403, 'Unable to determine user identity.');
    }
    
    /**
     * Get the current tenant_id from the request
     */
    private function getCurrentTenantId(Request $request)
    {
        return $request->attributes->get('tenant_id');
    }

    /**
     * Get unread notifications for the current user
     */
    public function getUnreadNotifications(Request $request)
    {
        $userIdentifier = $this->getCurrentUserIdentifier($request);
        $tenantId = $this->getCurrentTenantId($request);
        
        try {
            if ($userIdentifier['type'] === 'admin') {
                // Get conversations where admin is a participant and has unread messages
                $participantConversations = ConversationParticipant::where('admin_id', $userIdentifier['id'])
                    ->whereHas('conversation', function($query) use ($tenantId) {
                        $query->where('tenant_id', $tenantId);
                    })
                    ->with(['conversation.lastMessage.sender', 'conversation.lastMessage.senderAdmin'])
                    ->get();
            } else {
                // Get conversations where employee is a participant and has unread messages
                $participantConversations = ConversationParticipant::where('employee_id', $userIdentifier['id'])
                    ->whereHas('conversation', function($query) use ($tenantId) {
                        $query->where('tenant_id', $tenantId);
                    })
                    ->with(['conversation.lastMessage.sender', 'conversation.lastMessage.senderAdmin'])
                    ->get();
            }

            $notifications = [];
            
            foreach ($participantConversations as $participant) {
                $conversation = $participant->conversation;
                
                // Skip if no last message
                if (!$conversation->lastMessage) {
                    continue;
                }
                
                // Skip if the last message was sent by the current user
                if ($userIdentifier['type'] === 'admin' && 
                    $conversation->lastMessage->sender_admin_id == $userIdentifier['id']) {
                    continue;
                }
                
                if ($userIdentifier['type'] === 'employee' && 
                    $conversation->lastMessage->sender_id == $userIdentifier['id']) {
                    continue;
                }
                
                // Check if there are unread messages
                $unreadCount = $this->getUnreadMessageCount($participant, $conversation);
                
                if ($unreadCount > 0) {
                    // Get sender name
                    $senderName = $this->getSenderName($conversation->lastMessage);
                    
                    // Get conversation display name
                    $conversationName = $this->getConversationDisplayName($conversation, $userIdentifier);
                    
                    $notifications[] = [
                        'conversation_id' => $conversation->id,
                        'conversation_name' => $conversationName,
                        'sender_name' => $senderName,
                        'count' => $unreadCount,
                        'last_message_time' => $conversation->lastMessage->created_at,
                        'last_message_content' => substr($conversation->lastMessage->content, 0, 50) . '...'
                    ];
                }
            }
            
            // Sort by last message time, most recent first
            usort($notifications, function($a, $b) {
                return $b['last_message_time'] <=> $a['last_message_time'];
            });
            
            return response()->json($notifications);
            
        } catch (\Exception $e) {
            \Log::error('Error getting unread notifications: ' . $e->getMessage());
            return response()->json([]);
        }
    }
    
    /**
     * Get unread message count for a participant in a conversation
     */
    private function getUnreadMessageCount($participant, $conversation)
    {
        if (!$participant->last_read_at) {
            // If never read, count all messages in the conversation
            return $conversation->messages()->count();
        }
        
        // Count messages after last read time
        return $conversation->messages()
            ->where('created_at', '>', $participant->last_read_at)
            ->count();
    }
    
    /**
     * Get sender name from a message
     */
    private function getSenderName($message)
    {
        if ($message->sender_type === 'admin' && $message->senderAdmin) {
            return $message->senderAdmin->email;
        } elseif ($message->sender_type === 'employee' && $message->sender) {
            return $message->sender->name;
        }
        
        return $message->sender_type === 'admin' ? 'Admin' : 'Employee';
    }
    
    /**
     * Get conversation display name
     */
    private function getConversationDisplayName($conversation, $userIdentifier)
    {
        if ($conversation->type === 'group') {
            return $conversation->name;
        }
        
        // For personal chats, find the other participant
        $otherParticipant = $conversation->participants()
            ->where(function($query) use ($userIdentifier) {
                if ($userIdentifier['type'] === 'admin') {
                    $query->where(function($q) use ($userIdentifier) {
                        $q->whereNull('admin_id')
                          ->orWhere('admin_id', '!=', $userIdentifier['id']);
                    });
                } else {
                    $query->where(function($q) use ($userIdentifier) {
                        $q->whereNull('employee_id')
                          ->orWhere('employee_id', '!=', $userIdentifier['id']);
                    });
                }
            })
            ->with(['employee', 'admin'])
            ->first();

        if ($otherParticipant) {
            if ($otherParticipant->admin_id && $otherParticipant->admin) {
                return $otherParticipant->admin->email;
            } elseif ($otherParticipant->employee_id && $otherParticipant->employee) {
                return $otherParticipant->employee->name;
            }
        }

        return 'Personal Chat';
    }
    
    /**
     * Mark all notifications as read for the current user
     */
    public function markAllAsRead(Request $request)
    {
        $userIdentifier = $this->getCurrentUserIdentifier($request);
        $tenantId = $this->getCurrentTenantId($request);
        
        try {
            if ($userIdentifier['type'] === 'admin') {
                ConversationParticipant::where('admin_id', $userIdentifier['id'])
                    ->whereHas('conversation', function($query) use ($tenantId) {
                        $query->where('tenant_id', $tenantId);
                    })
                    ->update(['last_read_at' => now()]);
            } else {
                ConversationParticipant::where('employee_id', $userIdentifier['id'])
                    ->whereHas('conversation', function($query) use ($tenantId) {
                        $query->where('tenant_id', $tenantId);
                    })
                    ->update(['last_read_at' => now()]);
            }
            
            return response()->json(['success' => true]);
            
        } catch (\Exception $e) {
            \Log::error('Error marking all notifications as read: ' . $e->getMessage());
            return response()->json(['success' => false], 500);
        }
    }
    
    /**
     * Get total unread count for the current user
     */
    public function getUnreadCount(Request $request)
    {
        $userIdentifier = $this->getCurrentUserIdentifier($request);
        $tenantId = $this->getCurrentTenantId($request);
        
        try {
            $totalUnread = 0;
            
            if ($userIdentifier['type'] === 'admin') {
                $participants = ConversationParticipant::where('admin_id', $userIdentifier['id'])
                    ->whereHas('conversation', function($query) use ($tenantId) {
                        $query->where('tenant_id', $tenantId);
                    })
                    ->with('conversation')
                    ->get();
            } else {
                $participants = ConversationParticipant::where('employee_id', $userIdentifier['id'])
                    ->whereHas('conversation', function($query) use ($tenantId) {
                        $query->where('tenant_id', $tenantId);
                    })
                    ->with('conversation')
                    ->get();
            }
            
            foreach ($participants as $participant) {
                $totalUnread += $this->getUnreadMessageCount($participant, $participant->conversation);
            }
            
            return response()->json(['unread_count' => $totalUnread]);
            
        } catch (\Exception $e) {
            \Log::error('Error getting unread count: ' . $e->getMessage());
            return response()->json(['unread_count' => 0]);
        }
    }
}