<?php

namespace App\Http\Controllers;

use App\Models\Admin;
use App\Models\Conversation;
use App\Models\ConversationParticipant;
use App\Models\Employee;
use App\Models\Message;
use App\Services\ChatPushNotificationService;
use App\Services\NotificationService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;


class ChatController extends Controller
{
    private ChatPushNotificationService $pushService;
    private NotificationService $notificationService;
    public function __construct(ChatPushNotificationService $pushService, NotificationService $notificationService)
    {
        $this->pushService = $pushService;
        $this->notificationService = $notificationService;
    }
    /**
     * Get the current authenticated user's employee_id
     */
    private function getCurrentUserIdentifier(Request $request)
    {
        $user = $request->attributes->get('authenticated_user');
        
        if (get_class($user) === 'App\Models\Admin') {
            return [
                'type' => 'admin',
                'id' => $user->id,
                'user' => $user
            ];
        }
        
        // Handle employee
        if (isset($user->employee_id)) {
            return [
                'type' => 'employee',
                'id' => $user->employee_id,
                'user' => $user
            ];
        }
        
        abort(403, 'Unable to determine user identity.');
    }
    
    /**
     * Get the current tenant_id from the request
     */
    private function getCurrentTenantId(Request $request)
    {
        return $request->attributes->get('tenant_id');
    }
    
    /**
     * Display list of conversations
     */
    public function index(Request $request)
    {
        $userIdentifier = $this->getCurrentUserIdentifier($request);
        $tenantId = $this->getCurrentTenantId($request);
        
        // dd($userIdentifier['type']);
        if ($userIdentifier['type'] == 'admin') {
            // dd($userIdentifier['type']);
            // Admin can see all conversations in their tenant
            $conversations = Conversation::where('tenant_id', $tenantId)
                ->with(['lastMessage.sender', 'lastMessage.senderAdmin', 'participantEmployees', 'participantAdmins'])
                ->orderBy('last_message_at', 'desc')
                ->get()
                ->map(function ($conversation) use ($userIdentifier) {
                    $conversation->unread_count = $conversation->getUnreadCountForUser($userIdentifier['type'], $userIdentifier['id']);
                    return $conversation;
                });
        } else {
            // Employee sees only their conversations
            $employee = Employee::where('employee_id', $userIdentifier['id'])
                ->where('tenant_id', $tenantId)
                ->firstOrFail();
            // dd($employee);
            $conversations = $employee->conversations()
                ->with(['lastMessage.sender', 'lastMessage.senderAdmin', 'participantEmployees', 'participantAdmins'])
                ->orderBy('last_message_at', 'desc')
                ->get();
                // ->map(function ($conversation) use ($userIdentifier) {
                //     $conversation->unread_count = $conversation->getUnreadCountForUser($userIdentifier['type'], $userIdentifier['id']);
                //     return $conversation;
                // });
        }
        // dd($conversations);
        return view('chat.index', compact('conversations'));
    }

    /**
     * Show a specific conversation
     */
    public function show(Request $request, Conversation $conversation)
    {
        $tenantId = $this->getCurrentTenantId($request);
        // Check for AJAX request for new messages
        if ($request->ajax() && $request->has('last_message_id')) {
            return $this->checkNewMessages($request, $conversation);
        }
        
        $userIdentifier = $this->getCurrentUserIdentifier($request);
        
        // Get participant record based on user type
        if ($userIdentifier['type'] === 'admin') {
            // Check if admin is participant or create participant record
            $participant = $conversation->participants()
                ->where('admin_id', $userIdentifier['id'])
                ->first();
                
            if (!$participant) {
                // Auto-add admin as participant
                $participant = $conversation->participants()->create([
                    'admin_id' => $userIdentifier['id'],
                    'employee_id' => null,
                    'role' => 'admin',
                ]);
            }
        } else {
            $participant = $conversation->participants()
                ->where('employee_id', $userIdentifier['id'])
                ->firstOrFail();
        }

        // Get messages with sender info
        $messages = $conversation->messages()
            ->with(['sender', 'senderAdmin'])
            ->orderBy('created_at', 'asc')
            ->get();

        // Mark messages as read
        $participant->markAsRead();
 // Mark chat notifications as read when user opens conversation
        $this->notificationService->markChatNotificationsAsRead(
            $tenantId,
            $userIdentifier['type'],
            $userIdentifier['id'],
            $conversation->id
        );
        // Get conversation participants
        $participants = $conversation->getAllParticipants();

        return view('chat.show', compact('conversation', 'messages', 'participants'));
    }

    /**
     * Create a personal chat
     */
     public function createPersonalChat(Request $request)
    {
        // 1️⃣ Validate that both hidden fields are present:
        $data = $request->validate([
            'recipient_type' => ['required', 'in:admin,employee'],
            'recipient_id'   => ['required', 'integer'],
        ]);

        $recipientType = $data['recipient_type']; // "admin" or "employee"
        $recipientId   = $data['recipient_id'];   // e.g. 25

        // 2️⃣ Get current tenant ID and current user:
        //    (Assuming you have helper methods on your controller for these.)
        $tenantId    = $this->getCurrentTenantId($request);
        $currentUser = $this->getCurrentUserIdentifier($request);
        //    $currentUser is an array like ['type' => 'admin'|'employee', 'id' => 7]

        // 3️⃣ Verify that the recipient actually exists under this tenant.
        if ($recipientType === 'employee') {
            $recipient = Employee::where('employee_id', $recipientId)
                                 ->where('tenant_id', $tenantId)
                                 ->first();
        } else { // 'admin'
            $recipient = Admin::where('id', $recipientId)
                              ->where('tenant_id', $tenantId)
                              ->first();
        }

        if (! $recipient) {
            return back()->withErrors(['recipient' => ucfirst($recipientType) . ' not found or not in your tenant.']);
        }

        // 4️⃣ Prevent chatting with yourself:
        if ($currentUser['type'] === $recipientType && $currentUser['id'] == $recipientId) {
            return back()->withErrors(['recipient' => 'You cannot start a chat with yourself.']);
        }

        // 5️⃣ Check if a personal conversation already exists between these two participants:
        $existingConversation = $this->findExistingPersonalChat(
            $currentUser,
            ['type' => $recipientType, 'id' => $recipientId],
            $tenantId
        );

        if ($existingConversation) {
            // If found, redirect straight to it.
            return redirect()->route('chat.show', $existingConversation->id);
        }

        // 6️⃣ Otherwise, create a brand‐new personal conversation inside a transaction:
        DB::transaction(function () use ($currentUser, $recipientType, $recipientId, $tenantId, &$conversation) {
            // a) Create the conversation
            $conversation = Conversation::create([
                'tenant_id' => $tenantId,
                'type'      => 'personal',
            ]);

            // b) Add current user as participant
            $conversation->participants()->create([
                'admin_id'    => ($currentUser['type'] === 'admin' ? $currentUser['id'] : null),
                'employee_id' => ($currentUser['type'] === 'employee' ? $currentUser['id'] : null),
                'role'        => 'member',
            ]);
            // c) Add the chosen recipient as participant
            $conversation->participants()->create([
                'admin_id'    => ($recipientType === 'admin' ? $recipientId : null),
                'employee_id' => ($recipientType === 'employee' ? $recipientId : null),
                'role'        => 'member',
            ]);
        });

        // 7️⃣ Redirect to the newly‐created conversation:
        return redirect()->route('chat.show', $conversation->id);
    }

    /**
     * Find existing personal chat between two users
     */
    private function findExistingPersonalChat($user1, $user2, $tenantId)
    {
        $query = Conversation::where('tenant_id', $tenantId)
            ->where('type', 'personal');
            
        // Add conditions for first user
        if ($user1['type'] === 'admin') {
            $query->whereHas('participants', function ($q) use ($user1) {
                $q->where('admin_id', $user1['id']);
            });
        } else {
            $query->whereHas('participants', function ($q) use ($user1) {
                $q->where('employee_id', $user1['id']);
            });
        }
        
        // Add conditions for second user
        if ($user2['type'] === 'admin') {
            $query->whereHas('participants', function ($q) use ($user2) {
                $q->where('admin_id', $user2['id']);
            });
        } else {
            $query->whereHas('participants', function ($q) use ($user2) {
                $q->where('employee_id', $user2['id']);
            });
        }
        
        return $query->first();
    }

    /**
     * Create a group chat
     */
    public function createGroupChat(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string|max:500',
            'employee_ids' => 'required|array|min:1',
            'employee_ids.*' => 'exists:employees,employee_id'
        ]);

        $userIdentifier = $this->getCurrentUserIdentifier($request);
        $tenantId = $this->getCurrentTenantId($request);

        // Verify all selected employees belong to the same tenant
        $validEmployees = Employee::whereIn('employee_id', $request->employee_ids)
            ->where('tenant_id', $tenantId)
            ->pluck('employee_id')
            ->toArray();

        if (count($validEmployees) !== count($request->employee_ids)) {
            return back()->withErrors(['employee_ids' => 'Some selected employees are not from your organization.']);
        }

        $conversation = DB::transaction(function () use ($request, $userIdentifier, $tenantId, $validEmployees) {
            // Create the conversation
            $conversationData = [
                'tenant_id' => $tenantId,
                'type' => 'group',
                'name' => $request->name,
                'description' => $request->description,
            ];
            
            $conversation = Conversation::create($conversationData);

            // Add creator as admin based on their type
            if ($userIdentifier['type'] === 'admin') {
                $conversation->participants()->create([
                    'employee_id' => null,
                    'admin_id' => $userIdentifier['id'],
                    'role' => 'admin',
                ]);
            } else {
                $conversation->participants()->create([
                    'employee_id' => $userIdentifier['id'],
                    'admin_id' => null,
                    'role' => 'admin',
                ]);
            }

            // Add other participants (employees from the form)
            foreach ($validEmployees as $employeeId) {
                // Skip if it's the creator (only if creator is an employee)
                if ($userIdentifier['type'] === 'employee' && $employeeId == $userIdentifier['id']) {
                    continue;
                }
                
                $conversation->participants()->create([
                    'employee_id' => $employeeId,
                    'admin_id' => null,
                    'role' => 'member',
                ]);
            }

            return $conversation;
        });

        return redirect()->route('chat.show', $conversation);
    }
// Get display name for a specific user (handles both admin and employee)
public function getDisplayNameForUser($userType, $userId)
{
    if ($this->type === 'group') {
        return $this->name;
    }

    // For personal chats, find the other participant
    $otherParticipant = $this->participants()
        ->where(function($query) use ($userType, $userId) {
            if ($userType === 'admin') {
                // If current user is admin, find the other participant (not this admin)
                $query->where(function($q) use ($userId) {
                    $q->whereNull('admin_id')
                      ->orWhere('admin_id', '!=', $userId);
                });
            } else {
                // If current user is employee, find the other participant (not this employee)
                $query->where(function($q) use ($userId) {
                    $q->whereNull('employee_id')
                      ->orWhere('employee_id', '!=', $userId);
                });
            }
        })
        ->with(['employee', 'admin'])
        ->first();

    if ($otherParticipant) {
        if ($otherParticipant->admin_id) {
            return $otherParticipant->admin ? $otherParticipant->admin->display_name : 'Admin';
        } else {
            return $otherParticipant->employee ? $otherParticipant->employee->name : 'Employee';
        }
    }

    return 'Personal Chat';
}

// Get the other participant in a personal chat
public function getOtherParticipant($currentUserType, $currentUserId)
{
    if ($this->type === 'group') {
        return null;
    }

    $otherParticipant = $this->participants()
        ->where(function($query) use ($currentUserType, $currentUserId) {
            if ($currentUserType === 'admin') {
                $query->where(function($q) use ($currentUserId) {
                    $q->whereNull('admin_id')
                      ->orWhere('admin_id', '!=', $currentUserId);
                });
            } else {
                $query->where(function($q) use ($currentUserId) {
                    $q->whereNull('employee_id')
                      ->orWhere('employee_id', '!=', $currentUserId);
                });
            }
        })
        ->with(['employee', 'admin'])
        ->first();

    if ($otherParticipant) {
        if ($otherParticipant->admin_id) {
            return $otherParticipant->admin;
        } else {
            return $otherParticipant->employee;
        }
    }

    return null;
}

// Get participant type (for determining if it's admin or employee)
public function getOtherParticipantType($currentUserType, $currentUserId)
{
    $otherParticipant = $this->getOtherParticipant($currentUserType, $currentUserId);
    
    if (!$otherParticipant) {
        return null;
    }
    
    return $otherParticipant instanceof \App\Models\Admin ? 'admin' : 'employee';
}
    /**
     * Send a message
     */
   /**
     * Send a message
     */
    public function sendMessage(Request $request, Conversation $conversation)
    {
        $request->validate([
            'content' => 'required_without:attachments|string|max:5000',
            'attachments.*' => 'file|max:10240' // 10MB max
        ]);
        
        $userIdentifier = $this->getCurrentUserIdentifier($request);
        
        // Verify participant
        if ($userIdentifier['type'] === 'admin') {
            $participant = $conversation->participants()
                ->where('admin_id', $userIdentifier['id'])
                ->first();
                
            // Auto-add admin if not participant
            if (!$participant) {
                $participant = $conversation->participants()->create([
                    'admin_id' => $userIdentifier['id'],
                    'employee_id' => null,
                    'role' => 'member',
                ]);
            }
        } else {
            $participant = $conversation->participants()
                ->where('employee_id', $userIdentifier['id'])
                ->firstOrFail();
        }

        // Prepare message data based on sender type
        $messageData = [
            'conversation_id' => $conversation->id,
            'sender_type' => $userIdentifier['type'],
            'content' => $request->content ?? '',
            'type' => 'text',
        ];

        // Set the appropriate sender ID field
        if ($userIdentifier['type'] == 'admin') {
            $messageData['sender_admin_id'] = $userIdentifier['id'];
            $messageData['sender_id'] = null; // Explicitly set to null
        } else {
            $messageData['sender_id'] = $userIdentifier['id'];
            $messageData['sender_admin_id'] = null; // Explicitly set to null
        }

        // Handle attachments
        if ($request->hasFile('attachments')) {
            $attachments = [];
            foreach ($request->file('attachments') as $file) {
                $path = $file->store('chat-attachments/' . $conversation->id, 'public');
                $attachments[] = [
                    'path' => $path,
                    'name' => $file->getClientOriginalName(),
                    'size' => $file->getSize(),
                    'mime' => $file->getMimeType(),
                ];
            }
            $messageData['attachments'] = $attachments;
            $messageData['type'] = 'file';
        }

        $message = Message::create($messageData);
        // Update conversation last message time
        $conversation->update(['last_message_at' => now()]);

         try {
            $this->notificationService->sendChatNotification($message);
            Log::info('Chat notification sent', [
                'message_id' => $message->id,
                'conversation_id' => $conversation->id,
            ]);
        } catch (\Exception $e) {
            Log::error('Failed to send chat notification', [
                'message_id' => $message->id,
                'error' => $e->getMessage(),
            ]);
        }
        // Send push notifications to OTHER participants (not the sender)
        if ($message) {
            try {
                $this->pushService->sendNewMessageNotification($message);
                Log::info('Push notification sent for message', [
                    'message_id' => $message->id,
                    'conversation_id' => $conversation->id,
                    'sender_type' => $userIdentifier['type'],
                    'sender_id' => $userIdentifier['id']
                ]);
            } catch (\Exception $e) {
                Log::error('Failed to send push notification', [
                    'message_id' => $message->id,
                    'error' => $e->getMessage(),
                    'trace' => $e->getTraceAsString()
                ]);
                // Don't fail the message sending if push notification fails
            }
        }
        
        // Mark as read for sender
        $participant->markAsRead();

        if ($request->ajax()) {
            return response()->json([
                'message' => $message->load(['sender', 'senderAdmin']),
                'success' => true
            ]);
        }

        return redirect()->route('chat.show', $conversation)->with('success', 'Message sent successfully');
    }

    /**
     * Mark conversation as read
     */
    public function markAsRead(Request $request, Conversation $conversation)
    {
        $userIdentifier = $this->getCurrentUserIdentifier($request);
        
        if ($userIdentifier['type'] === 'admin') {
            $participant = $conversation->participants()
                ->where('admin_id', $userIdentifier['id'])
                ->first();
        } else {
            $participant = $conversation->participants()
                ->where('employee_id', $userIdentifier['id'])
                ->first();
        }

        if ($participant) {
            $participant->markAsRead();
        }

        return response()->json(['success' => true]);
    }

    /**
     * Search for conversations and employees
     */
    public function search(Request $request)
    {
        $request->validate([
            'query' => 'required|string|min:2'
        ]);

        $userIdentifier = $this->getCurrentUserIdentifier($request);
        $tenantId = $this->getCurrentTenantId($request);
        $query = $request->input('query');

        $results = [];

        // Search employees
        $employees = Employee::where('tenant_id', $tenantId)
            ->where(function ($q) use ($query) {
                $q->where('name', 'like', "%{$query}%")
                  ->orWhere('email', 'like', "%{$query}%");
            })
            ->limit(10)
            ->get()
            ->map(function ($employee) {
                return [
                    'type' => 'employee',
                    'id' => $employee->employee_id,
                    'name' => $employee->name,
                    'email' => $employee->email,
                ];
            });

        // Search admins
        $admins = Admin::where('tenant_id', $tenantId)
            ->where(function ($q) use ($query) {
                $q->where('email', 'like', "%{$query}%");
            })
            ->limit(10)
            ->get()
            ->map(function ($admin) {
                return [
                    'type' => 'admin',
                    'id' => $admin->id,
                    'name' => 'Admin: ' . $admin->email,
                    'email' => $admin->email,
                ];
            });

        $results['users'] = $employees->concat($admins);

        // Search conversations
        if ($userIdentifier['type'] === 'admin') {
            // Admin can search all conversations
            $conversations = Conversation::where('tenant_id', $tenantId)
                ->where(function ($q) use ($query) {
                    $q->where('name', 'like', "%{$query}%")
                      ->orWhereHas('messages', function ($mq) use ($query) {
                          $mq->where('content', 'like', "%{$query}%");
                      });
                })
                ->limit(10)
                ->get();
        } else {
            // Employee can only search their conversations
            $employee = Employee::where('employee_id', $userIdentifier['id'])
                ->where('tenant_id', $tenantId)
                ->first();
                
            if ($employee) {
                $conversations = $employee->conversations()
                    ->where(function ($q) use ($query) {
                        $q->where('name', 'like', "%{$query}%")
                          ->orWhereHas('messages', function ($mq) use ($query) {
                              $mq->where('content', 'like', "%{$query}%");
                          });
                    })
                    ->limit(10)
                    ->get();
            } else {
                $conversations = collect();
            }
        }

        $results['conversations'] = $conversations;

        return response()->json($results);
    }

    /**
     * Add participants to a group conversation
     */
    public function addParticipants(Request $request, Conversation $conversation)
    {
        $request->validate([
            'participants' => 'required|array|min:1',
            'participants.*.type' => 'required|in:employee,admin',
            'participants.*.id' => 'required'
        ]);

        $userIdentifier = $this->getCurrentUserIdentifier($request);
        
        // Check if user is admin of the group
        if ($userIdentifier['type'] === 'admin') {
            $participant = $conversation->participants()
                ->where('admin_id', $userIdentifier['id'])
                ->first();
        } else {
            $participant = $conversation->participants()
                ->where('employee_id', $userIdentifier['id'])
                ->first();
        }
            
        if (!$participant || $participant->role !== 'admin') {
            // Check if user is the creator
            if (!($conversation->created_by_type === $userIdentifier['type'] && 
                  $conversation->created_by_id == $userIdentifier['id'])) {
                return response()->json(['error' => 'You do not have permission to add members to this group.'], 403);
            }
        }

        $tenantId = $this->getCurrentTenantId($request);
        $added = 0;

        foreach ($request->participants as $newParticipant) {
            // Verify participant exists
            if ($newParticipant['type'] === 'employee') {
                $exists = Employee::where('employee_id', $newParticipant['id'])
                    ->where('tenant_id', $tenantId)
                    ->exists();
            } else {
                $exists = Admin::where('id', $newParticipant['id'])
                    ->where('tenant_id', $tenantId)
                    ->exists();
            }
            
            if (!$exists) {
                continue;
            }

            // Check if not already a participant
            if ($newParticipant['type'] === 'admin') {
                $alreadyExists = $conversation->participants()
                    ->where('admin_id', $newParticipant['id'])
                    ->exists();
            } else {
                $alreadyExists = $conversation->participants()
                    ->where('employee_id', $newParticipant['id'])
                    ->exists();
            }
                
            if (!$alreadyExists) {
                if ($newParticipant['type'] === 'admin') {
                    $conversation->participants()->create([
                        'admin_id' => $newParticipant['id'],
                        'employee_id' => null,
                        'role' => 'member',
                    ]);
                } else {
                    $conversation->participants()->create([
                        'employee_id' => $newParticipant['id'],
                        'admin_id' => null,
                        'role' => 'member',
                    ]);
                }
                $added++;
            }
        }

        // Add system message about new members
        if ($added > 0) {
            $messageData = [
                'conversation_id' => $conversation->id,
                'sender_type' => $userIdentifier['type'],
                'content' => "Added $added new member(s) to the group",
                'type' => 'system',
            ];
            
            if ($userIdentifier['type'] === 'admin') {
                $messageData['sender_admin_id'] = $userIdentifier['id'];
                $messageData['sender_id'] = null;
            } else {
                $messageData['sender_id'] = $userIdentifier['id'];
                $messageData['sender_admin_id'] = null;
            }
            
            Message::create($messageData);
            $conversation->update(['last_message_at' => now()]);
        }

        return response()->json([
            'success' => true,
            'message' => "$added member(s) added successfully"
        ]);
    }

    /**
     * Remove a participant from group conversation
     */
    public function removeParticipant(Request $request, Conversation $conversation, $participantType, $participantId)
    {
        $userIdentifier = $this->getCurrentUserIdentifier($request);
        
        // Check if user is admin of the group or removing themselves
        if ($userIdentifier['type'] === 'admin') {
            $participant = $conversation->participants()
                ->where('admin_id', $userIdentifier['id'])
                ->first();
        } else {
            $participant = $conversation->participants()
                ->where('employee_id', $userIdentifier['id'])
                ->first();
        }
            
        if (!$participant) {
            return response()->json(['error' => 'You are not a member of this group.'], 403);
        }
        
        // Check if user has permission to remove
        $isRemovingSelf = ($userIdentifier['type'] === $participantType && $userIdentifier['id'] == $participantId);
        if (!$isRemovingSelf && $participant->role !== 'admin') {
            return response()->json(['error' => 'Only admins can remove other members.'], 403);
        }

        // Remove the participant
        if ($participantType === 'admin') {
            $removed = $conversation->participants()
                ->where('admin_id', $participantId)
                ->delete();
        } else {
            $removed = $conversation->participants()
                ->where('employee_id', $participantId)
                ->delete();
        }

        if ($removed) {
            // Get participant name
            if ($participantType === 'admin') {
                $removedUser = Admin::find($participantId);
                $removedName = $removedUser ? $removedUser->email : 'Admin';
            } else {
                $removedUser = Employee::where('employee_id', $participantId)->first();
                $removedName = $removedUser ? $removedUser->name : 'Employee';
            }
            
            // Add system message
            $message = $isRemovingSelf 
                ? "{$removedName} left the group"
                : "{$removedName} was removed from the group";
                
            $messageData = [
                'conversation_id' => $conversation->id,
                'sender_type' => $userIdentifier['type'],
                'content' => $message,
                'type' => 'system',
            ];
            
            if ($userIdentifier['type'] === 'admin') {
                $messageData['sender_admin_id'] = $userIdentifier['id'];
                $messageData['sender_id'] = null;
            } else {
                $messageData['sender_id'] = $userIdentifier['id'];
                $messageData['sender_admin_id'] = null;
            }
            
            Message::create($messageData);
            $conversation->update(['last_message_at' => now()]);
        }

        return response()->json(['success' => true]);
    }

    /**
     * Leave a group conversation
     */
    public function leaveConversation(Request $request, Conversation $conversation)
    {
        $userIdentifier = $this->getCurrentUserIdentifier($request);
        
        // Remove from participants
        if ($userIdentifier['type'] === 'admin') {
            $removed = $conversation->participants()
                ->where('admin_id', $userIdentifier['id'])
                ->delete();
        } else {
            $removed = $conversation->participants()
                ->where('employee_id', $userIdentifier['id'])
                ->delete();
        }

        if ($removed) {
            // Get user name
            if ($userIdentifier['type'] === 'admin') {
                $user = Admin::find($userIdentifier['id']);
                $userName = $user ? $user->email : 'Admin';
            } else {
                $user = Employee::where('employee_id', $userIdentifier['id'])->first();
                $userName = $user ? $user->name : 'Employee';
            }
            
            // Add system message
            $messageData = [
                'conversation_id' => $conversation->id,
                'sender_type' => $userIdentifier['type'],
                'content' => "{$userName} left the group",
                'type' => 'system',
            ];
            
            if ($userIdentifier['type'] === 'admin') {
                $messageData['sender_admin_id'] = $userIdentifier['id'];
                $messageData['sender_id'] = null;
            } else {
                $messageData['sender_id'] = $userIdentifier['id'];
                $messageData['sender_admin_id'] = null;
            }
            
            Message::create($messageData);
            $conversation->update(['last_message_at' => now()]);
        }

        return response()->json(['success' => true]);
    }

    /**
     * Update conversation details (group only)
     */
    public function updateConversation(Request $request, Conversation $conversation)
    {
        if ($conversation->type !== 'group') {
            return response()->json(['error' => 'Cannot update personal conversations.'], 400);
        }

        $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string|max:500',
        ]);

        $userIdentifier = $this->getCurrentUserIdentifier($request);
        
        // Check if user is admin
        if ($userIdentifier['type'] === 'admin') {
            $participant = $conversation->participants()
                ->where('admin_id', $userIdentifier['id'])
                ->first();
        } else {
            $participant = $conversation->participants()
                ->where('employee_id', $userIdentifier['id'])
                ->first();
        }
            
        if (!$participant || $participant->role !== 'admin') {
            return response()->json(['error' => 'Only admins can update group details.'], 403);
        }

        $conversation->update([
            'name' => $request->name,
            'description' => $request->description,
        ]);

        return response()->json(['success' => true]);
    }

    /**
     * Edit a message
     */
    public function editMessage(Request $request, Message $message)
    {
        $request->validate([
            'content' => 'required|string|max:5000'
        ]);

        $userIdentifier = $this->getCurrentUserIdentifier($request);
        
        // Check if user is the sender
        if ($userIdentifier['type'] === 'admin') {
            if ($message->sender_admin_id !== $userIdentifier['id']) {
                return response()->json(['error' => 'You can only edit your own messages.'], 403);
            }
        } else {
            if ($message->sender_id !== $userIdentifier['id']) {
                return response()->json(['error' => 'You can only edit your own messages.'], 403);
            }
        }

        // Check if message is not too old (e.g., 24 hours)
        if ($message->created_at->diffInHours(now()) > 24) {
            return response()->json(['error' => 'Message is too old to edit.'], 400);
        }

        $message->update([
            'content' => $request->content,
            'is_edited' => true,
            'edited_at' => now(),
        ]);

        return response()->json([
            'success' => true,
            'message' => $message->load(['sender', 'senderAdmin'])
        ]);
    }

    /**
     * Delete a message
     */
    public function deleteMessage(Request $request, Message $message)
    {
        $userIdentifier = $this->getCurrentUserIdentifier($request);
        
        // Check if user is the sender
        if ($userIdentifier['type'] === 'admin') {
            if ($message->sender_admin_id !== $userIdentifier['id']) {
                return response()->json(['error' => 'You can only delete your own messages.'], 403);
            }
        } else {
            if ($message->sender_id !== $userIdentifier['id']) {
                return response()->json(['error' => 'You can only delete your own messages.'], 403);
            }
        }

        // Soft delete by replacing content
        $message->update([
            'content' => 'This message was deleted',
            'type' => 'deleted',
            'attachments' => null,
        ]);

        return response()->json(['success' => true]);
    }

    /**
     * AJAX: Check for new messages
     */
    private function checkNewMessages(Request $request, Conversation $conversation)
    {
        $lastMessageId = $request->get('last_message_id', 0);
        $userIdentifier = $this->getCurrentUserIdentifier($request);
        
        // Verify participant
        if ($userIdentifier['type'] === 'admin') {
            $participant = $conversation->participants()
                ->where('admin_id', $userIdentifier['id'])
                ->first();
        } else {
            $participant = $conversation->participants()
                ->where('employee_id', $userIdentifier['id'])
                ->first();
        }
            
        if (!$participant) {
            return response()->json(['error' => 'Unauthorized'], 403);
        }

        $newMessages = $conversation->messages()
            ->with(['sender', 'senderAdmin'])
            ->where('id', '>', $lastMessageId)
            ->orderBy('created_at', 'asc')
            ->get();

        return response()->json([
            'new_messages' => $newMessages,
            'count' => $newMessages->count()
        ]);
    }

    /**
     * Admin: List all conversations in the tenant
     */
    public function adminListAllConversations(Request $request)
    {
        $tenantId = $this->getCurrentTenantId($request);
        
        $conversations = Conversation::where('tenant_id', $tenantId)
            ->with(['participants.employee', 'participants.admin', 'lastMessage'])
            ->orderBy('last_message_at', 'desc')
            ->paginate(20);

        return view('chat.admin.conversations', compact('conversations'));
    }

    /**
     * Admin: View participants of a conversation
     */
    public function adminViewParticipants(Request $request, Conversation $conversation)
    {
        $tenantId = $this->getCurrentTenantId($request);
        
        if ($conversation->tenant_id !== $tenantId) {
            abort(403);
        }

        $participants = $conversation->participants()
            ->with(['employee', 'admin'])
            ->get();

        return view('chat.admin.participants', compact('conversation', 'participants'));
    }

    /**
     * Admin: Delete a conversation
     */
    public function adminDeleteConversation(Request $request, Conversation $conversation)
    {
        $tenantId = $this->getCurrentTenantId($request);
        
        if ($conversation->tenant_id !== $tenantId) {
            return response()->json(['error' => 'Unauthorized'], 403);
        }

        // This will cascade delete all messages and participants
        $conversation->delete();

        return response()->json(['success' => true]);
    }

    /**
     * Admin: Get chat statistics
     */
    public function adminChatStatistics(Request $request)
    {
        $tenantId = $this->getCurrentTenantId($request);
        
        $stats = [
            'total_conversations' => Conversation::where('tenant_id', $tenantId)->count(),
            'personal_chats' => Conversation::where('tenant_id', $tenantId)->where('type', 'personal')->count(),
            'group_chats' => Conversation::where('tenant_id', $tenantId)->where('type', 'group')->count(),
            'total_messages' => Message::whereHas('conversation', function($q) use ($tenantId) {
                $q->where('tenant_id', $tenantId);
            })->count(),
            'active_users' => ConversationParticipant::whereHas('conversation', function($q) use ($tenantId) {
                $q->where('tenant_id', $tenantId);
            })->select(DB::raw('DISTINCT COALESCE(employee_id, admin_id) as user_id'))->count(),
            'messages_today' => Message::whereHas('conversation', function($q) use ($tenantId) {
                $q->where('tenant_id', $tenantId);
            })->whereDate('created_at', today())->count(),
        ];

        return view('chat.admin.statistics', compact('stats'));
    }

    /**
     * Admin: Export chat data
     */
    public function adminExportChatData(Request $request)
    {
        $request->validate([
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
            'conversation_id' => 'nullable|exists:conversations,id',
        ]);

        $tenantId = $this->getCurrentTenantId($request);
        
        $query = Message::with(['sender', 'senderAdmin', 'conversation'])
            ->whereHas('conversation', function($q) use ($tenantId) {
                $q->where('tenant_id', $tenantId);
            });

        if ($request->start_date) {
            $query->whereDate('created_at', '>=', $request->start_date);
        }

        if ($request->end_date) {
            $query->whereDate('created_at', '<=', $request->end_date);
        }

        if ($request->conversation_id) {
            $query->where('conversation_id', $request->conversation_id);
        }

        $messages = $query->orderBy('created_at')->get();

        // Generate CSV
        $csvData = "Date,Time,Conversation,Sender,Message,Type\n";
        foreach ($messages as $message) {
            $senderName = $message->sender_type === 'admin' 
                ? ($message->senderAdmin ? $message->senderAdmin->email : 'Admin')
                : ($message->sender ? $message->sender->name : 'Employee');
                
            $csvData .= sprintf(
                "%s,%s,%s,%s,%s,%s\n",
                $message->created_at->format('Y-m-d'),
                $message->created_at->format('H:i:s'),
                $message->conversation->name ?? 'Personal Chat',
                $senderName,
                str_replace(["\n", "\r", ","], [' ', ' ', ';'], $message->content),
                $message->type
            );
        }

        return response($csvData)
            ->header('Content-Type', 'text/csv')
            ->header('Content-Disposition', 'attachment; filename="chat-export-' . now()->format('Y-m-d') . '.csv"');
    }
}