<?php

namespace App\Http\Controllers;

use App\Models\BusinessCard;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Intervention\Image\Facades\Image;

class BusinessCardController extends Controller
{
    /**
     * Get the current authenticated user's information
     */
    private function getCurrentUserIdentifier(Request $request)
    {
        $user = $request->attributes->get('authenticated_user');

        if (get_class($user) === 'App\Models\Admin') {
            return [
                'type' => 'admin',
                'id' => $user->id,
                'user' => $user
            ];
        }

        // Handle employee
        if (isset($user->employee_id)) {
            return [
                'type' => 'employee',
                'id' => $user->employee_id,
                'user' => $user
            ];
        }

        abort(403, 'Unable to determine user identity.');
    }

    /**
     * Get the current tenant_id from the request
     */
    private function getCurrentTenantId(Request $request)
    {
        return $request->attributes->get('tenant_id');
    }

    /**
     * Get current user using Auth facade (fallback method)
     */
    private function getCurrentUser()
    {
        if (Auth::guard('web')->check()) {
            return Auth::guard('web')->user();
        } elseif (Auth::guard('admin')->check()) {
            return Auth::guard('admin')->user();
        } elseif (Auth::guard('manager')->check()) {
            return Auth::guard('manager')->user();
        } elseif (Auth::guard('employee')->check()) {
            return Auth::guard('employee')->user();
        } elseif (Auth::guard('points_admin')->check()) {
            return Auth::guard('points_admin')->user();
        }

        return null;
    }

    /**
     * Get current user type
     */
    private function getCurrentUserType()
    {
        if (Auth::guard('employee')->check()) {
            return 'employee';
        } elseif (Auth::guard('admin')->check() || Auth::guard('manager')->check() || Auth::guard('points_admin')->check()) {
            return 'admin';
        }

        return 'employee'; // default
    }

    /**
     * Display a listing of business cards for the current user
     */
    public function index(Request $request)
    {
        $userIdentifier = $this->getCurrentUserIdentifier($request);
        $tenantId = $this->getCurrentTenantId($request);
        
        $businessCards = BusinessCard::where('tenant_id', $tenantId)
            ->where('user_id', $userIdentifier['user']->id)
            ->where('user_type', $userIdentifier['type'])
            ->orderBy('created_at', 'desc')
            ->get();

        return view('business-cards.index', compact('businessCards'));
    }

    /**
     * Show the form for creating a new business card
     */
    public function create(Request $request)
    {
        $userIdentifier = $this->getCurrentUserIdentifier($request);
        $user = $userIdentifier['user'];

        // Pre-fill some data from user profile
        $defaultData = [
            'name' => $user->name ?? '',
            'email' => $user->email ?? '',
            'phone' => $user->phn ?? $user->phone ?? '',
            'company_name' => session('tenant_name') ?? 'Your Dealership',
            'location' => $user->city ?? '',
        ];

        // Available templates
        $templates = $this->getAvailableTemplates();

        // Available specialties for car dealership
        $availableSpecialties = [
            'New Car Sales',
            'Used Car Sales',
            'Auto Service',
            'Parts & Accessories',
            'Finance & Insurance',
            'Auto Body Repair',
            'Fleet Management',
            'Luxury Vehicles',
            'Commercial Vehicles',
            'Motorcycle Sales'
        ];

        // Available car brands
        $availableBrands = [
            'Toyota', 'Honda', 'Ford', 'Chevrolet', 'Nissan', 'BMW', 'Mercedes-Benz', 'Audi',
            'Volkswagen', 'Hyundai', 'Kia', 'Mazda', 'Subaru', 'Lexus', 'Acura', 'Infiniti',
            'Cadillac', 'Lincoln', 'Jeep', 'Ram', 'Dodge', 'Chrysler', 'Buick', 'GMC',
            'Volvo', 'Jaguar', 'Land Rover', 'Porsche', 'Tesla', 'Genesis', 'Alfa Romeo',
            'Maserati', 'Ferrari', 'Lamborghini'
        ];

        // Available certifications
        $availableCertifications = [
            'ASE Certified Technician',
            'Master ASE Certified',
            'Manufacturer Certified Sales',
            'Finance & Insurance Certified',
            'Customer Service Excellence',
            'Automotive Sales Professional',
            'Service Advisor Certified',
            'Parts Specialist Certified',
            'Auto Body Repair Certified',
            'Hybrid/Electric Vehicle Certified'
        ];

        return view('business-cards.create', compact(
            'defaultData',
            'templates',
            'availableSpecialties',
            'availableBrands',
            'availableCertifications'
        ));
    }

    /**
     * Store a newly created business card
     */
    public function store(Request $request)
    {
        $userIdentifier = $this->getCurrentUserIdentifier($request);
        $tenantId = $this->getCurrentTenantId($request);
        
        $request->validate([
            'name' => 'required|string|max:255',
            'title' => 'required|string|max:255',
            'company_name' => 'required|string|max:255',
            'phone' => 'required|string|max:20',
            'email' => 'required|email|max:255',
            'website' => 'nullable|url|max:255',
            'bio' => 'nullable|string|max:500',
            'specialties' => 'nullable|array',
            'brands_worked_with' => 'nullable|array',
            'certifications' => 'nullable|array',
            'years_experience' => 'nullable|integer|min:0|max:50',
            'languages_spoken' => 'nullable|string|max:255',
            'services_offered' => 'nullable|string|max:1000',
            'location' => 'nullable|string|max:255',
            'preferred_contact_method' => 'nullable|string|max:50',
            'profile_photo' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
            'company_logo' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
            'background_image' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
            'whatsapp' => 'nullable|string|max:20',
            'linkedin' => 'nullable|url|max:255',
            'facebook' => 'nullable|url|max:255',
            'instagram' => 'nullable|url|max:255',
            'twitter' => 'nullable|url|max:255',
            'template_id' => 'required|integer|between:1,5',
            'primary_color' => 'nullable|string|max:7',
            'secondary_color' => 'nullable|string|max:7',
            'font_style' => 'nullable|string|in:modern,classic,elegant',
            'additional_notes' => 'nullable|string|max:1000',
            'business_hours' => 'nullable|array'
        ]);

        $businessCardData = $request->except(['profile_photo', 'company_logo', 'background_image', 'primary_color', 'secondary_color']);

        $businessCardData['tenant_id'] = $tenantId;
        $businessCardData['user_id'] = $userIdentifier['user']->id;
        $businessCardData['user_type'] = $userIdentifier['type'];
        $businessCardData['unique_slug'] = Str::slug($request->name . '-' . Str::random(8));

        // Handle custom colors
        if ($request->primary_color || $request->secondary_color) {
            $businessCardData['custom_colors'] = json_encode([
                'primary' => $request->primary_color ?? '#007bff',
                'secondary' => $request->secondary_color ?? '#6c757d'
            ]);
        }

        // Handle file uploads
        if ($request->hasFile('profile_photo')) {
            $businessCardData['profile_photo'] = $this->handleImageUpload($request->file('profile_photo'), 'profile');
        }

        if ($request->hasFile('company_logo')) {
            $businessCardData['company_logo'] = $this->handleImageUpload($request->file('company_logo'), 'logo');
        }

        if ($request->hasFile('background_image')) {
            $businessCardData['background_image'] = $this->handleImageUpload($request->file('background_image'), 'background');
        }

        $businessCard = BusinessCard::create($businessCardData);

        return redirect()->route('business-cards.show', $businessCard->id)
            ->with('success', 'Business card created successfully!');
    }

    /**
     * Display the specified business card
     */
    public function show(Request $request, BusinessCard $businessCard)
    {
        $userIdentifier = $this->getCurrentUserIdentifier($request);

        // Check if user owns this business card
        if (!$this->canUserEditCard($businessCard, $userIdentifier)) {
            abort(403, 'Unauthorized access to business card.');
        }

        return view('business-cards.show', compact('businessCard'));
    }

    /**
     * Show the form for editing the specified business card
     */
    public function edit(Request $request, BusinessCard $businessCard)
    {
        $userIdentifier = $this->getCurrentUserIdentifier($request);

        // Check if user owns this business card
        if (!$this->canUserEditCard($businessCard, $userIdentifier)) {
            abort(403, 'Unauthorized access to business card.');
        }

        // Available templates
        $templates = $this->getAvailableTemplates();

        // Available options (same as create)
        $availableSpecialties = [
            'New Car Sales', 'Used Car Sales', 'Auto Service', 'Parts & Accessories',
            'Finance & Insurance', 'Auto Body Repair', 'Fleet Management', 'Luxury Vehicles',
            'Commercial Vehicles', 'Motorcycle Sales'
        ];

        $availableBrands = [
            'Toyota', 'Honda', 'Ford', 'Chevrolet', 'Nissan', 'BMW', 'Mercedes-Benz', 'Audi',
            'Volkswagen', 'Hyundai', 'Kia', 'Mazda', 'Subaru', 'Lexus', 'Acura', 'Infiniti',
            'Cadillac', 'Lincoln', 'Jeep', 'Ram', 'Dodge', 'Chrysler', 'Buick', 'GMC',
            'Volvo', 'Jaguar', 'Land Rover', 'Porsche', 'Tesla', 'Genesis', 'Alfa Romeo',
            'Maserati', 'Ferrari', 'Lamborghini'
        ];

        $availableCertifications = [
            'ASE Certified Technician', 'Master ASE Certified', 'Manufacturer Certified Sales',
            'Finance & Insurance Certified', 'Customer Service Excellence', 'Automotive Sales Professional',
            'Service Advisor Certified', 'Parts Specialist Certified', 'Auto Body Repair Certified',
            'Hybrid/Electric Vehicle Certified'
        ];

        return view('business-cards.edit', compact(
            'businessCard',
            'templates',
            'availableSpecialties',
            'availableBrands',
            'availableCertifications'
        ));
    }

    /**
     * Update the specified business card
     */
    public function update(Request $request, BusinessCard $businessCard)
    {
        $userIdentifier = $this->getCurrentUserIdentifier($request);

        // Check if user owns this business card
        if (!$this->canUserEditCard($businessCard, $userIdentifier)) {
            abort(403, 'Unauthorized access to business card.');
        }

        $request->validate([
            'name' => 'required|string|max:255',
            'title' => 'required|string|max:255',
            'company_name' => 'required|string|max:255',
            'phone' => 'required|string|max:20',
            'email' => 'required|email|max:255',
            'website' => 'nullable|url|max:255',
            'bio' => 'nullable|string|max:500',
            'specialties' => 'nullable|array',
            'brands_worked_with' => 'nullable|array',
            'certifications' => 'nullable|array',
            'years_experience' => 'nullable|integer|min:0|max:50',
            'languages_spoken' => 'nullable|string|max:255',
            'services_offered' => 'nullable|string|max:1000',
            'location' => 'nullable|string|max:255',
            'preferred_contact_method' => 'nullable|string|max:50',
            'profile_photo' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
            'company_logo' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
            'background_image' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
            'whatsapp' => 'nullable|string|max:20',
            'linkedin' => 'nullable|url|max:255',
            'facebook' => 'nullable|url|max:255',
            'instagram' => 'nullable|url|max:255',
            'twitter' => 'nullable|url|max:255',
            'template_id' => 'required|integer|between:1,5',
            'primary_color' => 'nullable|string|max:7',
            'secondary_color' => 'nullable|string|max:7',
            'font_style' => 'nullable|string|in:modern,classic,elegant',
            'additional_notes' => 'nullable|string|max:1000',
            'business_hours' => 'nullable|array',
            'is_active' => 'boolean',
            'is_public' => 'boolean'
        ]);

        $businessCardData = $request->except(['profile_photo', 'company_logo', 'background_image', 'primary_color', 'secondary_color']);

        // Handle custom colors
        if ($request->primary_color || $request->secondary_color) {
            $businessCardData['custom_colors'] = json_encode([
                'primary' => $request->primary_color ?? '#007bff',
                'secondary' => $request->secondary_color ?? '#6c757d'
            ]);
        }

        // Handle file uploads
        if ($request->hasFile('profile_photo')) {
            // Delete old photo if exists
            if ($businessCard->profile_photo && file_exists(public_path($businessCard->profile_photo))) {
                unlink(public_path($businessCard->profile_photo));
            }
            $businessCardData['profile_photo'] = $this->handleImageUpload($request->file('profile_photo'), 'profile');
        }

        if ($request->hasFile('company_logo')) {
            // Delete old logo if exists
            if ($businessCard->company_logo && file_exists(public_path($businessCard->company_logo))) {
                unlink(public_path($businessCard->company_logo));
            }
            $businessCardData['company_logo'] = $this->handleImageUpload($request->file('company_logo'), 'logo');
        }

        if ($request->hasFile('background_image')) {
            // Delete old background if exists
            if ($businessCard->background_image && file_exists(public_path($businessCard->background_image))) {
                unlink(public_path($businessCard->background_image));
            }
            $businessCardData['background_image'] = $this->handleImageUpload($request->file('background_image'), 'background');
        }

        $businessCard->update($businessCardData);

        return redirect()->route('business-cards.show', $businessCard->id)
            ->with('success', 'Business card updated successfully!');
    }

    /**
     * Remove the specified business card
     */
    public function destroy(Request $request, BusinessCard $businessCard)
    {
        $userIdentifier = $this->getCurrentUserIdentifier($request);

        // Check if user owns this business card
        if (!$this->canUserEditCard($businessCard, $userIdentifier)) {
            abort(403, 'Unauthorized access to business card.');
        }

        // Delete associated files
        if ($businessCard->profile_photo && file_exists(public_path($businessCard->profile_photo))) {
            unlink(public_path($businessCard->profile_photo));
        }

        if ($businessCard->company_logo && file_exists(public_path($businessCard->company_logo))) {
            unlink(public_path($businessCard->company_logo));
        }

        if ($businessCard->background_image && file_exists(public_path($businessCard->background_image))) {
            unlink(public_path($businessCard->background_image));
        }

        $businessCard->delete();

        return redirect()->route('business-cards.index')
            ->with('success', 'Business card deleted successfully!');
    }

    /**
     * Public view of business card (no auth required)
     */
    public function publicView($slug)
    {
        $businessCard = BusinessCard::where('unique_slug', $slug)
            ->where('is_active', true)
            ->where('is_public', true)
            ->firstOrFail();

        // Increment view count if method exists
        if (method_exists($businessCard, 'incrementViewCount')) {
            $businessCard->incrementViewCount();
        }

        return view('business-cards.public', compact('businessCard'));
    }

    /**
     * Toggle business card status (active/inactive)
     */
    public function toggleStatus(Request $request, BusinessCard $businessCard)
    {
        $userIdentifier = $this->getCurrentUserIdentifier($request);

        // Check if user owns this business card
        if (!$this->canUserEditCard($businessCard, $userIdentifier)) {
            abort(403, 'Unauthorized access to business card.');
        }

        $businessCard->update([
            'is_active' => !$businessCard->is_active
        ]);

        $status = $businessCard->is_active ? 'activated' : 'deactivated';

        return response()->json([
            'success' => true,
            'message' => "Business card {$status} successfully!",
            'is_active' => $businessCard->is_active
        ]);
    }

    /**
     * Download business card as vCard
     */
    public function downloadVCard(Request $request, BusinessCard $businessCard)
    {
        $userIdentifier = $this->getCurrentUserIdentifier($request);

        // Check if user owns this business card or if it's public
        if (!$this->canUserEditCard($businessCard, $userIdentifier) && (!$businessCard->is_public || !$businessCard->is_active)) {
            abort(403, 'Unauthorized access to business card.');
        }

        $vcard = $this->generateVCard($businessCard);

        $fileName = Str::slug($businessCard->name) . '-business-card.vcf';

        return response($vcard)
            ->header('Content-Type', 'text/vcard')
            ->header('Content-Disposition', 'attachment; filename="' . $fileName . '"');
    }

    /**
     * Helper Methods
     */

    /**
     * Check if user can edit the business card
     */
    private function canUserEditCard(BusinessCard $businessCard, array $userIdentifier)
    {
        return $businessCard->user_id == $userIdentifier['user']->id && 
               $businessCard->user_type == $userIdentifier['type'];
    }

    private function handleImageUpload($file, $type)
    {
        $timestamp = now()->format('YmdHis');
        $filename = $type . '_' . $timestamp . '_' . Str::random(10) . '.' . $file->getClientOriginalExtension();
        $path = 'business-cards/' . $filename;

        // Ensure directory exists
        if (!is_dir(public_path('business-cards'))) {
            mkdir(public_path('business-cards'), 0755, true);
        }

        // Move file to public folder
        $file->move(public_path('business-cards'), $filename);

        return 'business-cards/' . $filename;
    }

    private function getAvailableTemplates()
    {
        return [
            1 => [
                'id' => 1,
                'name' => 'Modern Dealership',
                'description' => 'Clean, professional design perfect for all automotive professionals',
                'preview' => 'images/templates/template-1-preview.jpg'
            ],
            2 => [
                'id' => 2,
                'name' => 'Luxury Professional',
                'description' => 'Premium design for luxury brand specialists and high-end sales',
                'preview' => 'images/templates/template-2-preview.jpg'
            ],
            3 => [
                'id' => 3,
                'name' => 'Service Specialist',
                'description' => 'Technical-focused design for service advisors and technicians',
                'preview' => 'images/templates/template-3-preview.jpg'
            ],
            4 => [
                'id' => 4,
                'name' => 'Sales Champion',
                'description' => 'Dynamic design to showcase sales achievements and experience',
                'preview' => 'images/templates/template-4-preview.jpg'
            ],
            5 => [
                'id' => 5,
                'name' => 'Executive Elite',
                'description' => 'Executive-level design for managers and dealership owners',
                'preview' => 'images/templates/template-5-preview.jpg'
            ]
        ];
    }

    private function generateVCard(BusinessCard $businessCard)
    {
        $vcard = "BEGIN:VCARD\n";
        $vcard .= "VERSION:3.0\n";
        $vcard .= "FN:{$businessCard->name}\n";
        $vcard .= "ORG:{$businessCard->company_name}\n";
        $vcard .= "TITLE:{$businessCard->title}\n";
        $vcard .= "TEL:{$businessCard->phone}\n";
        $vcard .= "EMAIL:{$businessCard->email}\n";

        if ($businessCard->website) {
            $vcard .= "URL:{$businessCard->website}\n";
        }

        if ($businessCard->location) {
            $vcard .= "ADR:;;{$businessCard->location};;;;\n";
        }

        if ($businessCard->bio) {
            $vcard .= "NOTE:{$businessCard->bio}\n";
        }

        $vcard .= "END:VCARD\n";

        return $vcard;
    }
}