<?php

namespace App\Http\Controllers;

use App\Mail\OTPMail;
use App\Models\Admin;
use App\Models\Department;
use App\Models\Employee;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use App\Services\TenantManager;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Log;

class AuthController extends Controller
{
    // H-B: ALL Employees Login
    public function login()
    {
        return view('genericLogin');
    }

    // H-B: ALL Employees Login
    public function genAttemptLogin(Request $request)
    {
        $validate = $request->validate([
            'email' => "required | email",
            'password' => "required"
        ]);

        $emp = Employee::where('email', $request->email)
            ->where('status', '2')->first();
        try {
            if($emp)
            {
                if($emp->role == 'Employee')
                {
                    // Attempt login using Employee guard
                    $auth = Auth::guard('employee')->attempt([
                        'email' => $request->email,
                        'password' => $request->password]);
                    if ($auth && Hash::check($request->password, $emp->password)) {
                        session(['isemp' => $emp->toArray()]);
                        // Login successful
                        return response()->json([
                            'Error' => false,
                            'Message' => 'Logged in successfully...!',
                            'route' => 'emp.dashboard',
                        ]);
                    } else {
                        // Invalid credentials
                        return response()->json([
                            'Error' => true,
                            'Message' => 'Invalid Email or Password'
                        ]);
                    }
                }
                elseif ($emp->role == 'Manager')
                {
                    // Attempt login using Manager guard
                    $auth = Auth::guard('manager')->attempt([
                        'email' => $request->email,
                        'password' => $request->password]);
                    if ($auth && Hash::check($request->password, $emp->password)) {
                        session(['ismngr' => $emp->toArray()]);
                        // Login successful
                        return response()->json([
                            'Error' => false,
                            'Message' => 'Logged in successfully...!',
                            'route' => 'mngr.dashboard',
                        ]);
                    } else {
                        // Invalid credentials
                        return response()->json([
                            'Error' => true,
                            'Message' => 'Invalid Email or Password'
                        ]);
                    }
                }
                else if($emp->role == 'Admin')
                {
                    // Attempt login using admin guard
                    $auth = Auth::guard('pointsAdmin')->attempt([
                        'email' => $request->email,
                        'password' => $request->password
                    ]);

                    if ($auth) {
                        session(['isadmin' => $emp->toArray()]);
                        // Login successful
                        return response()->json([
                            'Error' => false,
                            'Message' => 'Logged in successfully...!',
                            'route' => 'pAdmin.dashboard',
                        ]);
                    } else {
                        // Invalid credentials
                        return response()->json([
                            'Error' => true,
                            'Message' => 'Invalid Email or Password'
                        ]);
                    }
                }
            }
            else
            {
                return response()->json([
                    'Error' => true,
                    'Message' => 'No Employee Found!'
                ]);
            }
        }
        catch (\Throwable $t) {
            // Handle any exception
            return response()->json([
                'Error' => true,
                'Message' => $t->getMessage()
            ], 422);
        }
    }

    public function superAdminLogin()
    {
        return view('SuperAdmin.auth.login');
    }

    // H-B : SuperAdmin Login
    public function attempt_login(Request $request)
    {
        // Get the tenant from TenantManager
        $tenantManager = app(TenantManager::class);
        $tenant = $tenantManager->getTenant();
        $tenantId = $tenant ? $tenant->id : null;
        
        $validate = $request->validate([
            'email' => "required | email",
            'password' => "required"
        ]);
        if ($tenantId) {
            $owner = Admin::where('email', $request->email)
                ->where('tenant_id', $tenantId)
                ->first();
        } else {
            $owner = Admin::where('email', $request->email)->first();
        }

        try {
            if ($owner) {
                // Attempt login using admin guard
                if (Auth::guard('admin')->attempt([
                    'email' => $request->email,
                    'password' => $request->password
                ])) {
                    // Create owner data
                    $ownerData = $owner->toArray();
                    
                    // CRITICAL CHANGE: Replace the admin ID with tenant ID in the session
                    // This allows all controllers to keep using $this->owner['id']
                    if ($tenantId) {
                        $ownerData['original_id'] = $ownerData['id']; // Keep original for reference if needed
                        $ownerData['id'] = $tenantId; // Set ID to tenant ID
                        $ownerData['tenant_id'] = $tenantId; // Also set tenant_id
                    }
                    // Store owner data in session
                    session(['owner' => $ownerData]);
                    
                    return response()->json([
                        'Error' => false,
                        'Message' => 'Logged in successfully...'
                    ]);
                } else {
                    return response()->json([
                        'Error' => true,
                        'Message' => 'Invalid Email or Password'
                    ]);
                }
            } else {
                return response()->json([
                    'Error' => true,
                    'Message' => 'No Admin Found!'
                ]);
            }
        } catch (\Throwable $t) {
            return response()->json([
                'Error' => true,
                'Message' => $t->getMessage()
            ], 422);
        }
    }

    public function forgot($role)
    {
        return view('SuperAdmin.auth.forgetPassword', compact('role'));
    }

    // H-B: Attempt Send OTP
    public function attemptForgot(Request $request)
    {
        $emp = Employee::where('email', $request->email)
            ->where('role', $request->role)->select('email', 'name')->first();

        if($emp)
        {
            $code = str_pad(rand(0, 99999), 5, '0', STR_PAD_LEFT);

            $check = DB::table('password_reset_tokens')->updateOrInsert(
                ['email' => $emp->email],
                [
                    'otp' => $code,
                    'expires_at' => now()->addMinutes(15), // OTP valid for 15 minutes
                    'created_at' => now(),
                ]
            );
            $email = $emp->email;
            Mail::to($email)->send(new OTPMail($code, $emp->name));

            return response()->json([
                'Error' => false,
                'Message' => 'Your OTP Code has been Sent to your registered email Successfully!'
            ]);
        }
        else
        {
            return response()->json([
                'Error' => true,
                'Message' => 'Email not Registered.'
            ]);
        }
    }

    public function otp($email)
    {
        return view('SuperAdmin.auth.otp', compact('email'));
    }

    // H-B: Verify Code
    public function verifyCode(Request $request)
    {
        $code = implode("", $request->code );
        $record = DB::table('password_reset_tokens')->where('email', $request->email)->first();

        if (!$record || $record->otp != $code) {
            return response()->json([
                'Error' => true,
                'Message' => 'Invalid OTP',
            ]);
        }
        if (Carbon::now()->greaterThan($record->expires_at)) {
            return response()->json([
                'Error' => true,
                'Message' => 'OTP is Expired',
            ]);
        }

        return response()->json([
            'Error' => false,
            'Message' => 'OTP Code Verified!',
        ]);
    }

    // H-B: Reset Password Page
    public function resetPassword($email)
    {
        return view('SuperAdmin.auth.resetPassword', compact('email'));
    }

    // H-B: Attempt Reset Password
    public function attemptResetPassword(Request $request)
    {
        if($request->newPass != $request->cnfrmPass)
        {
            return response()->json([
                'Error' => true,
                'Message' => 'Passwords Must be Same.'
            ]);
        }
        else
        {
            $emp = Employee::where('email', $request->email)->first();

            $emp->update([
                'password' => Hash::make($request->newPass),
            ]);

            if($emp)
            {
                return response()->json([
                    'Error' => false,
                    'role' => $emp->role,
                    'Message' => "Password Updated Successfully!"
                ]);
            }
            else
            {
                return response()->json([
                    'Error' => true,
                    'Message' => "Password Update Failed!"
                ]);
            }
        }
    }

    // H-B : Super Admin Logout
    public function logout()
    {
        Auth::guard('admin')->logout();
        return redirect(route('auth.login'));
    }

    // H-B: Manager Login Page
    public function mngrLogin()
    {
        return view('Manager.Auth.login');
    }

    // H-B : Manager Login Attempt
    public function mngrAttemptLogin(Request $request)
    {
        $validate = $request->validate([
            'email' => "required | email",
            'password' => "required"
        ]);

        $ismngr = Employee::where('email', $request->email)
            ->where('role', 'Manager')
            ->where('status', '2')
            ->first();
        try {

            if ($ismngr) {

                // Attempt login using admin guard
                $auth = Auth::guard('manager')->attempt([
                    'email' => $request->email,
                    'password' => $request->password]);
                if ($auth && Hash::check($request->password, $ismngr->password)) {
                    session(['ismngr' => $ismngr->toArray()]);
                    // Login successful
                    return response()->json([
                        'Error' => false,
                        'Message' => 'Logged in successfully...!',
                    ]);
                } else {
                    // Invalid credentials
                    return response()->json([
                        'Error' => true,
                        'Message' => 'Invalid Email or Password'
                    ]);
                }
            } else {
                return response()->json([
                    'Error' => true,
                    'Message' => 'No Manager Found!'
                ]);
            }

        } catch (\Throwable $t) {
            // Handle any exception
            return response()->json([
                'Error' => true,
                'Message' => $t->getMessage() . ' on line: ' . $t->getLine() . ' on file: ' . $t->getFile()
            ], 422);
        }
    }

    // H-B : Employee Logout
    public function mngrLogout()
    {
        Auth::guard('manager')->logout();
        return redirect(route('gen.login'));
    }

    // H-B : Employee Login Page
    public function empLogin()
    {
        return view('Employee.Auth.login');
    }

    // H-B : Employee Login Attempt
    public function empLoginAttempt(Request $request) {

        $validate = $request->validate([
            'email' => "required | email",
            'password' => "required"
        ]);

        $isemp = Employee::where('email', $request->email)
            ->where('role', 'Employee')
            ->where('status', '2')
            ->first();

        try {

            if ($isemp) {

                // Attempt login using admin guard
                $auth = Auth::guard('employee')->attempt([
                    'email' => $request->email,
                    'password' => $request->password]);
                if ($auth && Hash::check($request->password, $isemp->password)) {
                    session(['isemp' => $isemp->toArray()]);
                    // Login successful
                    return response()->json([
                        'Error' => false,
                        'Message' => 'Logged in successfully...!',
                    ]);
                } else {
                    // Invalid credentials
                    return response()->json([
                        'Error' => true,
                        'Message' => 'Invalid Email or Password'
                    ]);
                }
            } else {
                return response()->json([
                    'Error' => true,
                    'Message' => 'No Employee Found!'
                ]);
            }

        } catch (\Throwable $t) {
            // Handle any exception
            return response()->json([
                'Error' => true,
                'Message' => $t->getMessage() . ' on line: ' . $t->getLine() . ' on file: ' . $t->getFile()
            ], 422);
        }

    }

    // H-B : Employee Logout
    public function empLogout()
    {
        Auth::guard('employee')->logout();
        return redirect(route('gen.login'));
    }

    // H-B: Points Admin Login Page
    public function pAdminLogin()
    {
        return view('PointsAdmin.Auth.login');
    }

    // H-B : Points Admin Login Attempt
    public function pAdminLoginAttempt(Request $request)
    {
        $validate = $request->validate([
            'email' => "required | email",
            'password' => "required"
        ]);
        $isadmin = Employee::where('email', $request->email)
            ->where('role', 'Admin')
            ->where('status', '2')
            ->first();
        try {

            if ($isadmin) {
                // Attempt login using admin guard
                $auth = Auth::guard('pointsAdmin')->attempt([
                    'email' => $request->email,
                    'password' => $request->password
                ]);

                if ($auth) {
                    session(['isadmin' => $isadmin->toArray()]);
                    // Login successful
                    return response()->json([
                        'Error' => false,
                        'Message' => 'Logged in successfully...!',
                    ]);
                } else {
                    // Invalid credentials
                    return response()->json([
                        'Error' => true,
                        'Message' => 'Invalid Email or Password'
                    ]);
                }
            } else {
                return response()->json([
                    'Error' => true,
                    'Message' => 'No Admin Found!'
                ]);
            }

        } catch (\Throwable $t) {
            // Handle any exception
            return response()->json([
                'Error' => true,
                'Message' => $t->getMessage() . ' on line: ' . $t->getLine() . ' on file: ' . $t->getFile()
            ], 422);
        }

    }

    // H-B : Points Admin Logout
    public function pAdminLogout()
    {
        Auth::guard('pointsAdmin')->logout();
        return redirect(route('gen.login'));
    }

    // H-B: Owner Login
    public function ownerLogin()
    {
        return view ('Dealership.login');
    }

    // H-B: Owner Attempt Login
    public function ownerAttemptLogin(Request $request)
    {
        $validate = $request->validate([
            'email' => "required | email",
            'password' => "required"
        ]);

        $owner = User::where('email', $request->email)->first();

        try {

            if ($owner) {
                // Attempt login using admin guard
                if (Auth::guard('owner')->attempt([
                    'email' => $request->email,
                    'password' => $request->password
                ])
                ) {
                    session(['superOwner' => $owner->toArray()]);
                    // Login successful
                    return response()->json([
                        'Error' => false,
                        'Message' => 'Logged in successfully...'
                    ]);
                } else {
                    // Invalid credentials
                    return response()->json([
                        'Error' => true,
                        'Message' => 'Invalid Email or Password'
                    ]);
                }
            } else {
                return response()->json([
                    'Error' => true,
                    'Message' => 'No Owner Found!'
                ]);
            }

        } catch (\Throwable $t) {
            // Handle any exception
            return response()->json([
                'Error' => true,
                'Message' => $t->getMessage()
            ], 422);
        }
    }

    // H-B : Points Admin Logout
    public function ownerLogout()
    {
        Auth::guard('owner')->logout();
        return redirect(route('owner.login'));
    }

}
